import { Injectable, Logger } from '@nestjs/common';
import { AiService } from '../ai/ai.service.js';
import { SupabaseService } from '../supabase/supabase.service.js';
import { RerankService } from './rerank.service.js';
import { appConfig } from '../config/app.config.js';
import { assemblePromptSections, logRetrievalObservability } from './context-assembler.js';
import type { AssembledContext, ContextChunk, RetrievalTier } from './types/context.types.js';

/** Safely coerce Supabase query results (typed non-null but nullable at runtime) to an array. */
function asArray<T>(data: T[] | null | undefined): T[] {
  return data ?? [];
}

@Injectable()
export class ContextPipelineService {
  private readonly logger = new Logger(ContextPipelineService.name);

  constructor(
    private readonly aiService: AiService,
    private readonly supabaseService: SupabaseService,
    private readonly rerankService: RerankService,
  ) {}

  public async assembleContext(goalId: string, userId: string): Promise<AssembledContext> {
    const startTime = Date.now();
    const queryText = await this.buildQueryText(goalId);
    if (queryText.length === 0) {
      throw new Error(
        `No query text available for goal ${goalId}: neither goal profile nor goal data found`,
      );
    }
    const queryEmbedding = await this.aiService.generateEmbedding(queryText);
    const retrieval = await this.retrieveChunks(queryEmbedding, goalId, userId);
    const rerankResult = await this.rerankService.rerank(queryText, retrieval.chunks);
    const { tier, fallbackReason } = this.determineTier(retrieval, rerankResult);

    logRetrievalObservability(this.logger, {
      allCandidates: rerankResult.allCandidates,
      selected: rerankResult.selected,
      tier,
      fallbackReason,
      goalId,
      latencyMs: Date.now() - startTime,
    });

    return assemblePromptSections(rerankResult.selected);
  }

  private determineTier(
    retrieval: { usedSqlFallback: boolean; fallbackReason?: string },
    rerankResult: { rerankApplied: boolean; failureReason?: string },
  ): { tier: RetrievalTier; fallbackReason: string | undefined } {
    if (retrieval.usedSqlFallback) {
      return { tier: 'sql_fallback', fallbackReason: retrieval.fallbackReason };
    }
    if (rerankResult.rerankApplied) {
      return { tier: 'hnsw_reranked', fallbackReason: undefined };
    }
    return { tier: 'hnsw_only', fallbackReason: rerankResult.failureReason };
  }

  private async buildQueryText(goalId: string): Promise<string> {
    const supabase = this.supabaseService.getAdminClient();
    const { data: profile } = (await supabase
      .from('goal_profiles')
      .select('narrative_summary')
      .eq('goal_id', goalId)
      .single()) as { data: { narrative_summary: string } | null };

    if (typeof profile?.narrative_summary === 'string' && profile.narrative_summary.length > 0) {
      return profile.narrative_summary;
    }

    const { data: goal } = (await supabase
      .from('goals')
      .select('title, description')
      .eq('id', goalId)
      .single()) as { data: { title: string; description: string } | null };

    return `${goal?.title ?? ''} ${goal?.description ?? ''}`.trim();
  }

  private async retrieveChunks(
    queryEmbedding: number[],
    goalId: string,
    userId: string,
  ): Promise<{ chunks: ContextChunk[]; usedSqlFallback: boolean; fallbackReason?: string }> {
    try {
      return await this.hnswRetrieve(queryEmbedding, goalId, userId);
    } catch (error) {
      const reason = error instanceof Error ? error.message : String(error);
      this.logger.warn(`HNSW retrieval failed, falling back to SQL context stuffing: ${reason}`);
      const chunks = await this.sqlContextStuffing(goalId, userId);
      return { chunks, usedSqlFallback: true, fallbackReason: reason };
    }
  }

  private async hnswRetrieve(
    queryEmbedding: number[],
    goalId: string,
    userId: string,
  ): Promise<{ chunks: ContextChunk[]; usedSqlFallback: boolean }> {
    const supabase = this.supabaseService.getAdminClient();
    const { data, error } = await supabase.rpc('match_goal_context', {
      query_embedding: JSON.stringify(queryEmbedding),
      p_goal_id: goalId,
      p_user_id: userId,
      p_content_types: undefined as unknown as string[],
      match_threshold: appConfig.roadmap.matchThreshold,
      match_count: appConfig.roadmap.matchCount,
    });
    if (error) {
      throw error;
    }
    const rows = asArray(data);
    return {
      chunks: rows.map((row) => ({
        id: row.id,
        content_text: row.content_text,
        content_type: row.content_type,
        similarity: row.similarity,
        metadata: (row.metadata as Record<string, unknown>) ?? {},
      })),
      usedSqlFallback: false,
    };
  }

  private async sqlContextStuffing(goalId: string, userId: string): Promise<ContextChunk[]> {
    const supabase = this.supabaseService.getAdminClient();
    const { data, error } = await supabase
      .from('context_embeddings')
      .select('id, content_text, content_type, metadata')
      .eq('user_id', userId)
      .or(`goal_id.eq.${goalId},goal_id.is.null`)
      .order('created_at', { ascending: false });
    if (error) {
      throw error;
    }
    return asArray(data).map((row) => ({
      id: row.id,
      content_text: row.content_text,
      content_type: row.content_type,
      similarity: 0,
      metadata: (row.metadata as Record<string, unknown> | null) ?? {},
    }));
  }
}
