import { Injectable, Logger } from '@nestjs/common';
import { IntakePromptService } from './intake-prompt.service.js';
import { IntakeQualityService } from './intake-quality.service.js';
import { IntakeProfileService } from './intake-profile.service.js';
import { MAX_GENERATION_ATTEMPTS } from './constants/intake.constants.js';
import type { GeneratedQuestion, PriorBatchContext } from './intake-prompt.service.js';
import type { ProfileResult } from './types/intake.types.js';

export type BatchGenerationResult =
  | { kind: 'questions'; questions: GeneratedQuestion[] }
  | { kind: 'complete'; profileResult: ProfileResult }
  | { kind: 'fallback' };

interface GenerateParams {
  goalDescription: string;
  priorBatches: PriorBatchContext[];
  nextBatchNumber: number;
  userId: string;
  goalId: string;
}

@Injectable()
export class IntakeGenerationService {
  private readonly logger = new Logger(IntakeGenerationService.name);

  constructor(
    private readonly promptService: IntakePromptService,
    private readonly qualityService: IntakeQualityService,
    private readonly profileService: IntakeProfileService,
  ) {}

  public async generateBatch(params: GenerateParams): Promise<BatchGenerationResult> {
    for (let attempt = 1; attempt <= MAX_GENERATION_ATTEMPTS; attempt++) {
      const result = await this.tryGenerate(params, attempt);
      if (result !== null) {
        return result;
      }
    }
    return { kind: 'fallback' };
  }

  private async tryGenerate(
    params: GenerateParams,
    attempt: number,
  ): Promise<BatchGenerationResult | null> {
    try {
      const generated = await this.promptService.generateNextBatch(
        params.goalDescription,
        params.priorBatches,
        params.nextBatchNumber,
      );

      if (generated.is_complete) {
        const profileResult = await this.profileService.generateAndStoreProfile(
          params.userId,
          params.goalId,
          params.goalDescription,
        );
        return { kind: 'complete', profileResult };
      }

      const validation = this.qualityService.validateBatch(generated.questions);
      if (validation.valid) {
        return { kind: 'questions', questions: generated.questions };
      }

      const suffix = attempt === MAX_GENERATION_ATTEMPTS ? ' Serving fallback.' : ' Retrying...';
      this.logger.warn(
        `Batch attempt ${String(attempt)}/${String(MAX_GENERATION_ATTEMPTS)} validation failed (${validation.layer}): ${validation.errors.join(', ')}.${suffix}`,
      );
    } catch (error) {
      const suffix = attempt === MAX_GENERATION_ATTEMPTS ? ' Serving fallback.' : ' Retrying...';
      this.logger.warn(
        `Batch attempt ${String(attempt)}/${String(MAX_GENERATION_ATTEMPTS)} failed: ${error instanceof Error ? error.message : String(error)}.${suffix}`,
      );
    }
    return null;
  }
}
