# Momentum -- Backend

Momentum is an AI coaching backend that builds personalized development roadmaps through adaptive conversation. Rather than using static questionnaires, the system conducts a multi-batch intake dialogue where each round of questions is shaped by the user's previous answers. It then generates structured milestone plans grounded in retrieved context, with quality validated by an independent LLM judge.

## Tech Stack

| Layer         | Technology                                     |
|---------------|------------------------------------------------|
| Framework     | NestJS 11, TypeScript (strict mode)            |
| Database      | Supabase (Postgres + Auth + Row Level Security)|
| Vector Search | pgvector with HNSW indexing                    |
| LLM Routing   | OpenRouter (Gemini Flash default)              |
| Reranking     | Cohere Rerank v3.5                             |
| Voice         | Deepgram (Nova-3 STT, Aura-2 TTS)              |
| Validation    | class-validator + Zod-style structured output  |

## Architecture

### Module Map

```
src/
  ai/          -- OpenAI SDK wrapper: structured JSON generation + embeddings
  intake/      -- Adaptive question batching, answer quality scoring, profile generation
  roadmap/     -- Milestone/weekly/daily plan generation, check-ins, debriefs
  goal/        -- Goal CRUD and status lifecycle
  coach/       -- Coach persona management
  voice/       -- Speech-to-text and text-to-speech via Deepgram
  eval/        -- Offline evaluation suite with multi-persona gold standards
  supabase/    -- Admin + user-scoped Supabase clients
  config/      -- Static configuration, prompt templates
  common/      -- Auth guard, decorators
```

### Key Design Patterns

**Adaptive Intake** -- The intake module generates question batches dynamically. Each batch is informed by all prior answers, with an LLM evaluating answer quality to decide whether to probe deeper or move on. After 5-7 batches, a structured goal profile is synthesized from the full conversation.

**3-Tier Context Retrieval** -- Roadmap generation uses a retrieval pipeline with graceful degradation:
1. HNSW vector search (pgvector) + Cohere reranking -- the primary path
2. HNSW-only -- if the reranker is unavailable
3. SQL context stuffing -- fallback if vector search fails entirely

Each tier is logged with observability metadata (latency, candidate counts, fallback reasons).

**LLM-as-Judge Validation** -- Generated outputs (intake batches, roadmap milestones, weekly plans) are validated by a separate LLM call that scores quality dimensions and flags issues. This runs asynchronously via event listeners, so it does not block the user-facing response.

**Event-Driven Side Effects** -- `EventEmitter2` decouples generation from quality evaluation and embedding storage. Key events: `roadmap.generated`, `batch.answered`, `batch.served`, `profile.generated`.

### Goal Status Lifecycle

```
intake_in_progress -> intake_completed -> roadmap_generating -> active
```

### Evaluation Suite

An offline evaluation harness (`npm run eval`) tests intake quality across multiple simulated user personas against gold-standard baselines. An independent judge model scores each run, producing a comparative report.

## Running Locally

```bash
npm install
cp .env.example .env
# edit .env variables
npm run start:dev
```

### Environment Variables

| Variable                   | Purpose                        |
|----------------------------|--------------------------------|
| `SUPABASE_URL`             | Supabase project URL           |
| `SUPABASE_SERVICE_ROLE_KEY`| Service role key               |
| `OPENROUTER_API_KEY`       | LLM routing via OpenRouter     |
| `COHERE_API_KEY`           | Reranking API                  |
| `DEEPGRAM_API_KEY`         | Voice STT/TTS                  |
| `PORT`                     | Server port (default: 3000)    |

### Available Scripts

```bash
npm run start:dev      # Development server with hot reload
npm run build          # Compile TypeScript
npm run test           # Unit tests
npm run test:e2e       # End-to-end tests
npm run eval           # Run evaluation suite against gold standards
npm run lint           # ESLint with auto-fix
```

API documentation (Swagger) is served at `/docs` when the server is running.
